﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using HIPS.CommonSchemas;
using HIPS.Web.Components.ServiceModel;

using HIPS.Web.Components.Common;

namespace HIPS.Web.Data.Hips
{
    internal static class ResponseMessageExtensions
    {

        /// <summary>
        /// Returns a list of messages based on the contents of the HIPS response.
        /// If the Status is not OK a message either an error or a message describing the status will be returned.
        /// </summary>
        /// <param name="hipsResponse">HipsResponse to generate the list of messages from.</param>
        /// <returns>List containing messages generated from the HIPS response.</returns>
        /// <remarks>
        /// This is only general messages common to all responses.
        /// Individual requests may have to add additional messages beyond this method.
        /// </remarks>
        public static ResponseMessageList ToMessageListExpectOkStatus(this HipsResponse hipsResponse)
        {
            ResponseMessageList result = new ResponseMessageList();

            // Always return hips error messages as errors
            if (!string.IsNullOrEmpty(hipsResponse.HipsErrorMessage))
            {
                string description = "";
                if (hipsResponse.Status == HipsResponseIndicator.DemographicMismatchWarning)
                {
                    string[] mismatched = hipsResponse.HipsErrorMessage.Replace("[", "").Split(new string[] { "]" }, StringSplitOptions.RemoveEmptyEntries);
                    description = string.Format("This document and the local patient record have a {0}.", string.Join(" and ", mismatched));
                }
                else
                {
                    description = hipsResponse.HipsErrorMessage;
                }
                result.Add(string.Format("{0} (Status: {1})", description, hipsResponse.Status), MessageLevel.Error);
            }
            else if (hipsResponse.Status != HipsResponseIndicator.OK)
            {
                // Check if there is a response code description
                if (!string.IsNullOrEmpty(hipsResponse.ResponseCodeDescription))
                {
                    // Show the Response Code/Description as a Warning
                    result.Add(string.Format("{0} - {1}", hipsResponse.ResponseCode, hipsResponse.ResponseCodeDescription), MessageLevel.Warning);
                }
                else // If not, return the HIPS status
                {
                    // Show the HIPS status as Information
                    result.Add(string.Format("Unexpected status: {0}", hipsResponse.Status), MessageLevel.Information);
                }
            }
            
            return result;
        }
    }
}
